<?php

namespace Modules\Achat\Controllers;

use App\Controllers\BaseController;
use App\Libraries\DatatableQuery;
use App\Traits\CommonTrait;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Modules\Achat\Models\Achat as AchatModel;
use Modules\Administration\Traits\UserTrait;
use Modules\Params\Models\BrancheModel;
use Modules\Params\Models\ProvinceModel;
use Modules\Params\Models\StatutAchat;
use Modules\Params\Models\UniteAchat;
use Psr\Log\LoggerInterface;
use stdClass;
use UConverter;

use function PHPUnit\Framework\fileExists;

class Achat extends BaseController
{
    public AchatModel $achatMdl;
    public BrancheModel $brancheMdl;
    public ProvinceModel $provinceMdl;
    public StatutAchat $statutAchatMdl;
    public UniteAchat $uniteAchatMdl;

    protected string $viewPath;

    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);
        $this->achatMdl = new AchatModel();
        $this->brancheMdl = new BrancheModel();
        $this->provinceMdl = new ProvinceModel();
        $this->statutAchatMdl = new StatutAchat();
        $this->uniteAchatMdl = new UniteAchat();

        $this->viewPath = "Modules\\Achat\\Views\\";
    }


    public function index()
    {
        $this->viewData['url_list'] = "/achat/data";
        $this->viewData['js'] = "Common.js";
        $this->viewData['title'] = "Liste des achats";
        $this->viewData['provinces'] = CommonTrait::formatData($this->provinceMdl);
        $this->viewData['statuts'] = CommonTrait::formatData($this->statutAchatMdl);
        $this->viewData['unites'] = CommonTrait::formatData($this->uniteAchatMdl);
        $this->viewData['responsables'] = UserTrait::usersArrayFormat();

        return view($this->viewPath . 'achat/index', $this->viewData);
    }

    function fecth()
    {
        $search_column = ['id', 'province_id', 'branche_id', 'description', 'code_projet', 'code_donneur', 'code_activite', 'code_atlas', 'nom_projet', 'partenaire', 'unite_id', 'quantite', 'prix_unitaire', 'budget_total', 'budget_disponible', 'date_acquisition', 'date_lancement', 'nom_acheteur', 'procedures', 'date_depot', 'date_attribution', 'montant_marche', 'date_probable_livraison', 'lieu_livraison', 'responsable', 'statut_id'];
        $order_column = ['id', 'province_id', 'branche_id', 'description', 'code_projet', 'code_donneur', 'code_activite', 'code_atlas', 'nom_projet', 'partenaire', 'unite_id', 'quantite', 'prix_unitaire', 'budget_total', 'budget_disponible', 'date_acquisition', 'date_lancement', 'nom_acheteur', 'procedures', 'date_depot', 'date_attribution', 'montant_marche', 'date_probable_livraison', 'lieu_livraison', 'responsable', 'statut_id'];

        $achatMdl = $this->achatMdl;

        if (\auth()->user()->inGroup('provinceManager' || $this->request->getPost('province_id') > 0)) {
            $achatMdl->where('province_id', \auth()->user()->province_id);
        }

        $fetch_data = DatatableQuery::build_query($achatMdl, $search_column, $order_column);
        $output = $this->formatAchats($this->viewData['locale'], $fetch_data);

        return $this->response->setJSON($output);
    }

    public function add()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(4);

        $this->viewData['url_list'] = "";
        $this->viewData['js'] = "Achat.js";
        $this->viewData['route'] = "/achat/save";
        $this->viewData['achat'] =  $this->initAchat($id);
        $this->viewData['provinces'] = CommonTrait::formatData($this->provinceMdl);
        $this->viewData['statuts'] = CommonTrait::formatData($this->statutAchatMdl);
        $this->viewData['unites'] = CommonTrait::formatData($this->uniteAchatMdl);
        $this->viewData['responsables'] = UserTrait::usersArrayFormat();


        if ($id > 0)
            $title = "Editer un achat";
        else
            $title = "Nouveau achat";

        $this->viewData['title'] = $title;

        return view($this->viewPath . 'achat/add', $this->viewData);
    }

    public function save()
    {
        $id = $this->request->getPost('id');

        if (!$this->validate($this->_rules())) {

            $this->viewData['url_list'] = "/achat/data";
            $this->viewData['js'] = "Achat.js";
            $this->viewData['route'] = "/achat/save";
            $this->viewData['achat'] =  $this->initAchat($id);
            $this->viewData['provinces'] = CommonTrait::formatData($this->provinceMdl);
            $this->viewData['statuts'] = CommonTrait::formatData($this->statutAchatMdl);
            $this->viewData['unites'] = CommonTrait::formatData($this->uniteAchatMdl);
            $this->viewData['responsables'] = UserTrait::usersArrayFormat();

            if ($id > 0)
                $title = "Editer un achat";
            else
                $title = "Nouveau achat";

            $this->viewData['title'] = $title;


            return view($this->viewPath . 'achat/add', $this->viewData);
        } else {

            $unAchat = $this->unAchat();

            if ($id) {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>Un achat a été modifié</div>");
                $stmt = $this->achatMdl->update($id, $unAchat);
            } else {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>Un achat  a été bien enregistré</div>");
                $stmt = $this->achatMdl->save($unAchat);
            }

            if ($stmt) {
                return redirect()->to(site_url($this->viewData['locale'] . '/achat'));
            }
        }
    }


    public function delete()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(4);

        $stmnt = $this->achatMdl->delete($id);

        if ($stmnt) {
            $this->session->setFlashdata('message', "<div class='row alert alert-success'>Un achat a été suprimé</div>");
            return redirect()->to(site_url($this->viewData['locale'] . 'achat'));
        }
    }

    public function detail()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(4);

        $this->viewData['url_list'] = "/achat/data";
        $this->viewData['js'] = "Common.js";
        $this->viewData['title'] = "Détail  d'un achat";
        $this->viewData['achat'] = $this->initAchat($id);
        return view($this->viewPath . 'achat/detail', $this->viewData);
    }

    private function unAchat()
    {
        $achat = [];

        $achat['province_id'] = $this->request->getPost('province_id');
        $achat['branche_id'] = $this->request->getPost('branche_id');
        $achat['description'] = $this->request->getPost('description');
        $achat['code_projet'] = $this->request->getPost('code_projet');
        $achat['code_donneur'] = $this->request->getPost('code_donneur');
        $achat['code_activite'] = $this->request->getPost('code_activite');
        $achat['code_atlas'] = $this->request->getPost('code_atlas');
        $achat['nom_projet'] = $this->request->getPost('nom_projet');
        $achat['partenaire'] = $this->request->getPost('partenaire');
        $achat['unite_id'] = $this->request->getPost('unite_id');
        $achat['quantite'] = $this->request->getPost('quantite');
        $achat['prix_unitaire'] = $this->request->getPost('prix_unitaire');
        $achat['budget_total'] = $this->request->getPost('budget_total');
        $achat['budget_disponible'] = $this->request->getPost('budget_disponible');
        $achat['date_acquisition'] = $this->request->getPost('date_acquisition');
        $achat['date_lancement'] = $this->request->getPost('date_lancement');
        $achat['nom_acheteur'] = $this->request->getPost('nom_acheteur');
        $achat['procedures'] = $this->request->getPost('procedures');
        $achat['date_depot'] = $this->request->getPost('date_depot');
        $achat['date_attribution'] = $this->request->getPost('date_attribution');
        $achat['montant_marche'] = $this->request->getPost('montant_marche');
        $achat['date_probable_livraison'] = $this->request->getPost('date_probable_livraison');
        $achat['lieu_livraison'] = $this->request->getPost('lieu_livraison');
        $achat['responsable'] = $this->request->getPost('responsable');
        $achat['user_id'] = \auth()->user()->id;
        $achat['statut_id'] = $this->request->getPost('statut_id');

        return $achat;
    }

    public function initAchat($id = NULL)
    {
        $achat = NULL;
        $achat = $this->achatMdl->find($id);

        if (empty($achat)) {
            $achat = new stdClass();
            $achat->id = 0;
            $achat->province_id = 0;
            $achat->branche_id = 0;
            $achat->description = "";
            $achat->code_projet = '';
            $achat->code_donneur = '';
            $achat->code_activite = '';
            $achat->code_atlas = "";
            $achat->nom_projet = "";
            $achat->partenaire = "";
            $achat->unite_id = 0;
            $achat->quantite = '';
            $achat->prix_unitaire = 0;
            $achat->budget_total = 0;
            $achat->budget_disponible = 0;
            $achat->date_acquisition = "";
            $achat->date_lancement = "";
            $achat->nom_acheteur = "";
            $achat->procedures = "";
            $achat->date_depot = "";
            $achat->date_attribution = "";
            $achat->montant_marche = 0;
            $achat->date_probable_livraison = "";
            $achat->lieu_livraison = "";
            $achat->responsable = 0;
            $achat->statut_id = 0;
            $achat->user_id = 0;
            $achat->created_at = NULL;
        }

        return $achat;
    }

    private function _rules()
    {
        $rules = [];
        $rules['province_id'] = ['rules'  => 'required'];
        $rules['branche_id'] = ['rules'  => 'required'];
        $rules['description'] = ['rules'  => 'required'];
        $rules['code_projet'] = ['rules'  => 'required'];
        $rules['code_donneur'] = ['rules'  => 'required'];
        $rules['code_activite'] = ['rules'  => 'required'];
        $rules['code_atlas'] = ['rules'  => 'required'];
        $rules['nom_projet'] = ['rules'  => 'required'];
        $rules['partenaire'] = ['rules'  => 'required'];
        $rules['unite_id'] = ['rules'  => 'required'];
        $rules['quantite'] = ['rules'  => 'required'];
        $rules['prix_unitaire'] = ['rules'  => 'required'];
        $rules['budget_total'] = ['rules'  => 'required'];
        $rules['budget_disponible'] = ['rules'  => 'required'];
        $rules['date_acquisition'] = ['rules'  => 'required'];
        $rules['date_lancement'] = ['rules'  => 'required'];
        $rules['nom_acheteur'] = ['rules'  => 'required'];
        $rules['procedures'] = ['rules'  => 'required'];
        $rules['date_depot'] = ['rules'  => 'required'];
        $rules['date_attribution'] = ['rules'  => 'required'];
        $rules['montant_marche'] = ['rules'  => 'required'];
        $rules['date_probable_livraison'] = ['rules'  => 'required'];
        $rules['lieu_livraison'] = ['rules'  => 'required'];
        $rules['responsable'] = ['rules'  => 'required'];
        $rules['statut_id'] = ['rules'  => 'required'];

        return $rules;
    }



    private function formatAchats($locale, $datas)
    {
        $i = 1;
        $array_data = array();
        if (!empty($datas['data'])) {
            foreach ($datas['data'] as $data) {
                $responsable = $data->responsable > 0 ? UserTrait::user($data->responsable) : NULL;
                $responsableNom = $responsable ? $responsable->nom . " " . $responsable->prenom : "";

                $sub_array = array();

                $sub_array[] =  $i;
                // $sub_array[] =  $data->province_id > 0 ? CommonTrait::formatData($this->provinceMdl, $data->province_id) : "";
                $sub_array[] =  $data->description;
                $sub_array[] =  $data->code_projet;
                $sub_array[] =  $data->code_donneur;
                $sub_array[] =  $data->code_activite;
                // $sub_array[] =  $data->code_atlas;
                // $sub_array[] =  $data->nom_projet;
                $sub_array[] =  $data->unite_id > 0 ? CommonTrait::formatData($this->uniteAchatMdl, $data->unite_id) : "";
                $sub_array[] =  $data->quantite;
                $sub_array[] =  $data->prix_unitaire;
                // $sub_array[] =  $data->budget_total;
                // $sub_array[] =  $data->date_attribution;
                // $sub_array[] =  $data->date_probable_livraison;
                $sub_array[] =  $responsableNom;

                $option = anchor($locale . "/achat/add/" . $data->id, "<i class='fa fa-edit'></i>", ['class' => 'text-info']) . " "
                    . anchor($locale . "/achat/detail/" . $data->id, "<i class='fa fa-eye'></i>", ['class' => 'text-secondary']) . " ";
                if (\auth()->user()->inGroup('secretaire', 'admin') ||  \auth()->user()->hasPermission('achat.delete')) {
                    $option .= anchor($locale . "/achat/delete/" . $data->id, "<i class='fa fa-trash'></i>", ['class' => 'text-danger']);
                }
                $sub_array[] = $option;

                $array_data[] = $sub_array;
                $i++;
            }
        }

        return [
            "draw" => intval($_POST["draw"]),
            "recordsTotal" => $datas['recordsTotal'],
            "recordsFiltered" => $datas['recordsFiltered'],
            "data" => $array_data
        ];
    }

    function export()
    {
        $uri = current_url(true);
        $province_id = $uri->getSegment(5);

        if ($province_id > 0) {
            $province = CommonTrait::formatData($this->provinceMdl, $province_id);
        } else {
            $province = "Toutes les provinces";
        }

        $fileName = str_replace(' ', '', $province);
        $csv_file_name = "rapportExport/achat/achat_" . $fileName . ".csv";
        $file = fopen($csv_file_name, 'w');

        $arrays = [];
        $arrays[0] = ['', $this->formatText("Gestion des achats")];
        $arrays[1] = ['', $this->formatText($province)];
        $arrays[2] = [];
        $arrays[3] = [
            '#',
            $this->formatText('Province'),
            $this->formatText('Description du marché'),
            $this->formatText('Code Projet'),
            $this->formatText('Code donneur'),
            $this->formatText('Code Activité'),
            $this->formatText('Code Atlas'),
            $this->formatText('Nom du projet'),
            $this->formatText('Partenaire'),
            $this->formatText('Unité'),
            $this->formatText('Quantité'),
            $this->formatText('Prix unitaire'),
            $this->formatText('Budget total'),
            $this->formatText('Budget disponible'),
            $this->formatText("Date d'acquisition"),
            $this->formatText("Date de lancement"),
            $this->formatText("Nom de l'acheteur"),
            $this->formatText("Procédure de passation"),
            $this->formatText("Date de dépot"),
            $this->formatText("Date d'attribution"),
            $this->formatText("Montant du marché"),
            $this->formatText("Date probable de livraison"),
            $this->formatText("Lieu de livraison"),
            $this->formatText("Responsable"),
            $this->formatText("Status")

        ];

        if ($province_id > 0) {
            $achats = $this->achatMdl->where('province_id', $province_id)->findAll();
        } else {
            $achats = $this->achatMdl->findAll();
        }

        $i = 1;
        foreach ($achats as $achat) {
            $responsable = $achat->responsable > 0 ? UserTrait::user($achat->responsable) : NULL;
            $responsableNom = $responsable ? $responsable->nom . " " . $responsable->prenom : "";

            $sub_array = [];
            $sub_array[] =  $i;
            $sub_array[] =  $achat->province_id > 0 ? CommonTrait::formatData($this->provinceMdl, $achat->province_id) : "";
            $sub_array[] =  $achat->description;
            $sub_array[] =  $achat->code_projet;
            $sub_array[] =  $achat->code_donneur;
            $sub_array[] =  $achat->code_activite;
            $sub_array[] =  $achat->code_atlas;
            $sub_array[] =  $achat->nom_projet;
            $sub_array[] =  $achat->partenaire;
            $sub_array[] =  $achat->unite_id > 0 ? CommonTrait::formatData($this->uniteAchatMdl, $achat->unite_id) : "";
            $sub_array[] =  $achat->quantite;
            $sub_array[] =  $achat->prix_unitaire;
            $sub_array[] =  $achat->budget_total;
            $sub_array[] =  $achat->budget_disponible;
            $sub_array[] =  $achat->date_acquisition;
            $sub_array[] =  $achat->date_lancement;
            $sub_array[] =  $achat->nom_acheteur;
            $sub_array[] =  $achat->procedures;
            $sub_array[] =  $achat->date_depot;
            $sub_array[] =  $achat->date_attribution;
            $sub_array[] =  $achat->montant_marche;
            $sub_array[] =  $achat->date_probable_livraison;
            $sub_array[] =  $achat->lieu_livraison;
            $sub_array[] =  $responsableNom;
            $sub_array[] =  CommonTrait::formatData($this->statutAchatMdl, $achat->statut_id);

            $arrays[] = $sub_array;
            $i++;
        }


        foreach ($arrays as $array) {
            fputcsv($file, $array, ";");
        }
        fclose($file);
        $btn = anchor($csv_file_name, "Exporter un <i class='fa fa-file-csv'></i>", ['class' => 'text-success', 'target' => '_blank']);

        $status = 0;
        if (fileExists($csv_file_name)) {
            $status = 1;
        }

        return $this->response->setJSON(['status' => $status, 'btnExport' => $btn]);
    }

    function formatText($string = NULL)
    {
        if (gettype($string) == "string") {
            $string = UConverter::transcode($string, 'ISO-8859-1', 'UTF8');
        }

        if (gettype($string) == "array") {
            foreach ($string as $key => $value) {
                $string[$key] = UConverter::transcode($value, 'ISO-8859-1', 'UTF8');
            }
        }

        return $string;
    }
}
