<?php

namespace Modules\Volontaire\Controllers;

use App\Controllers\BaseController;
use App\Libraries\DatatableQuery;
use App\Models\Pays;
use App\ThirdParty\fpdf\PdfModel;
use App\Traits\CommonTrait;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Modules\Administration\Models\User;
use Modules\Params\Models\BrancheModel;
use Modules\Params\Models\CollineModel;
use Modules\Params\Models\CommuneModel;
use Modules\Params\Models\ProfessionModel;
use Modules\Params\Models\ZoneModel;
use Modules\Volontaire\Models\Volontaire as ModelsVolontaire;
use Psr\Log\LoggerInterface;
use stdClass;
use UConverter;

use function PHPUnit\Framework\fileExists;

class Volontaire extends BaseController
{
    protected ModelsVolontaire $volontaireModel;
    protected ProfessionModel $professionModel;
    protected BrancheModel $brancheModel;

    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);
        $this->volontaireModel = new ModelsVolontaire();
        $this->professionModel = new ProfessionModel();
        $this->brancheModel = new BrancheModel();
    }

    public function index()
    {
        $this->viewData['url_list'] = "";
        $this->viewData['js'] = "Volontaire.js";

        $volontaire = $this->initVolontaire(\auth()->user()->id);
        $this->viewData['volontaire'] = $volontaire;
        $this->viewData['genres'] = CommonTrait::genres();
        $this->viewData['pays'] = CommonTrait::formatData(new Pays());
        $this->viewData['professions'] = CommonTrait::formatData($this->professionModel);
        $this->viewData['estStaff'] = CommonTrait::estStaff();
        $this->viewData['ouiNon'] = CommonTrait::yesno();
        $this->viewData['branches'] = CommonTrait::formatData($this->brancheModel);

        return view('volonatire/identite', $this->viewData);
    }

    function list()
    {
        $this->viewData['url_list'] = "/volontaire/data";
        // $this->viewData['js'] = "Volontaire.js";
        $this->viewData['volontaires'] = $this->volontaireModel->findAll();
        $this->viewData['js'] = "Common.js";
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());

        return view('volonatire/index', $this->viewData);
    }

    function fecth()
    {
        $search_column = ['titre', 'nom', 'prenom', 'surnom', 'genre', 'email', 'telephone', 'dateNaissance', 'nationaliteId', 'professionId', 'fonction', 'employeur', 'estStaff', 'passportImagePath', 'userId', 'brancheId', 'communeId', 'zoneId', 'collineId'];
        $order_column = ['titre', 'nom', 'prenom', 'surnom', 'genre', 'email', 'telephone', 'dateNaissance', 'nationaliteId', 'professionId', 'fonction', 'employeur', 'estStaff', 'passportImagePath', 'userId', 'brancheId', 'communeId', 'zoneId', 'collineId'];

        $formationMdl = $this->volontaireModel;
        if (\auth()->user()->inGroup('brancheManager')) {
            $formationMdl->where('brancheId', \auth()->user()->brancheId);
        }

        if (\auth()->user()->inGroup('communeManager')) {
            $formationMdl->where('communeId', \auth()->user()->communeId);
        }

        if ($this->request->getPost('brancheId') > 0) {
            $formationMdl->where('brancheId', $this->request->getPost('brancheId'));
        }

        $fetch_data = DatatableQuery::build_query($formationMdl, $search_column, $order_column);
        $output = $this->formatVolontaires($this->viewData['locale'], $fetch_data);

        return $this->response->setJSON($output);
    }

    private function formatVolontaires($locale, $datas)
    {
        $i = 1;
        $array_data = array();
        if (!empty($datas['data'])) {
            foreach ($datas['data'] as $volontaire) {

                $sub_array = [];
                $sub_array[] = $i;
                $sub_array[] = $volontaire->titre . " " . $volontaire->nom . " " . $volontaire->prenom;
                $sub_array[] =  $volontaire->brancheId > 0 ? CommonTrait::formatData(new BrancheModel(), $volontaire->brancheId) : "";
                $sub_array[] =  $volontaire->communeId > 0 ? CommonTrait::formatData(new CommuneModel(), $volontaire->communeId) : "";
                // $sub_array[] = $volontaire->genre;
                $sub_array[] = $volontaire->telephone;
                $sub_array[] = $volontaire->email;
                // $sub_array[] = $volontaire->employeur;
                // $sub_array[] = $volontaire->estStaff;
                $sub_array[] = $volontaire->professionId > 0 ? CommonTrait::formatData(new ProfessionModel(), $volontaire->professionId) : "";
                $sub_array[] = "<a href='" . base_url($locale . '/volontaire/dash/' . $volontaire->id) . "'> Detail </a>";

                $array_data[] = $sub_array;
                $i++;
            }
        }

        return [
            "draw" => intval($_POST["draw"]),
            "recordsTotal" => $datas['recordsTotal'],
            "recordsFiltered" => $datas['recordsFiltered'],
            "data" => $array_data
        ];
    }

    public function save()
    {
        $id = $this->request->getPost('id');
        if (! $this->validate($this->valideForm())) {
            $this->viewData['url_list'] = "";
            $this->viewData['js'] = "Volontaire.js";

            $volontaire = $this->initVolontaire(\auth()->user()->id);
            $this->viewData['volontaire'] = $volontaire;
            $this->viewData['genres'] = CommonTrait::genres();
            $this->viewData['pays'] = CommonTrait::formatData(new Pays());
            $this->viewData['professions'] = CommonTrait::formatData($this->professionModel);
            $this->viewData['estStaff'] = CommonTrait::estStaff();
            $this->viewData['branches'] = CommonTrait::formatData($this->brancheModel);
            $this->viewData['ouiNon'] = CommonTrait::yesno();

            return view('volonatire/identite', $this->viewData);
        } else {
            // echo "<pre>";
            // print_r($this->volontaire());
            // echo "</pre>";

            $stmnt = $this->volontaireModel->update($id, $this->volontaire());

            //Update user table
            $userId = \auth()->user()->id;
            $userData = [
                'nom' => $this->request->getPost('nom'),
                'prenom' => $this->request->getPost('prenom'),
                'telephone' => $this->request->getPost('telephone'),
                'brancheId' => $this->request->getPost('brancheId'),
                'communeId' => $this->request->getPost('communeId'),
                'zoneId' => $this->request->getPost('zoneId'),
                'collineId' => $this->request->getPost('collineId')
            ];

            (new User())->update($userId, $userData);
            if ($stmnt) {
                $this->session->setFlashdata('message', "<alert class='ml-2 alert alert-success'>Vos informations de base ont mise a jour avec succès</alert>");
                return redirect()->to(site_url($this->viewData['locale'] . '/volontaire/identite'));
            }
        }
    }

    private function valideFormPhoto()
    {
        $rules = [];
        $rules['passportImagePath'] = [
            'uploaded[passportImagePath]',
            'max_size[passportImagePath,2024]',
            'is_image[passportImagePath]'
        ];

        return $rules;
    }

    public function photo()
    {
        $id = $this->request->getPost('id');

        if (!$this->validate($this->valideFormPhoto())) {
            $this->viewData['url_list'] = "";
            $this->viewData['js'] = "";

            $volontaire = $this->initVolontaire(\auth()->user()->id);
            $this->viewData['volontaire'] = $volontaire;
            $this->viewData['genres'] = CommonTrait::genres();
            $this->viewData['pays'] = CommonTrait::formatData(new Pays());
            $this->viewData['professions'] = CommonTrait::formatData($this->professionModel);
            $this->viewData['estStaff'] = CommonTrait::estStaff();

            return view('volonatire/identite', $this->viewData);
        } else {

            $file = $this->request->getFile('passportImagePath');

            if ($file->isValid() && !$file->hasMoved()) {
                $newName = $id . "_" . $file->getRandomName();
                $file->move('uploads/passport/', $newName);

                $stmnt = $this->volontaireModel->update($id, ['passportImagePath' => $newName]);
                if ($stmnt) {
                    $session = session();
                    $session->set('passportImage', $newName);

                    $this->session->setFlashdata('message', "<alert class='ml-2 alert alert-success'>Votre photo de profil a été chargée avec succès</alert>");
                    return redirect()->to(site_url($this->viewData['locale'] . '/volontaire/identite'));
                }
            }
        }
    }

    private function volontaire(): array
    {
        $volontaire = [];
        $volontaire['titre'] = $this->request->getPost('titre');
        $volontaire['nom'] = $this->request->getPost('nom');
        $volontaire['prenom'] = $this->request->getPost('prenom');
        $volontaire['surnom'] = $this->request->getPost('surnom');
        $volontaire['genre'] = $this->request->getPost('genre');
        $volontaire['email'] = $this->request->getPost('email');
        $volontaire['telephone'] = $this->request->getPost('telephone');
        $volontaire['dateNaissance'] = $this->request->getPost('dateNaissance');
        $volontaire['nationaliteId'] = $this->request->getPost('nationaliteId');
        $volontaire['professionId'] = $this->request->getPost('professionId');
        $volontaire['fonction'] = $this->request->getPost('fonction');
        $volontaire['employeur'] = $this->request->getPost('employeur');
        $volontaire['estStaff'] = $this->request->getPost('estStaff');
        $volontaire['brancheId'] = $this->request->getPost('brancheId');
        $volontaire['communeId'] = $this->request->getPost('communeId');
        $volontaire['zoneId'] = $this->request->getPost('zoneId');
        $volontaire['collineId'] = $this->request->getPost('collineId');
        $volontaire['estSectionJeunesse'] = $this->request->getPost('estSectionJeunesse');
        $volontaire['sectionJeunesse'] = $this->request->getPost('sectionJeunesse');
        $volontaire['userId'] = \auth()->user()->id;

        return $volontaire;
    }

    private function initVolontaire($id = NULL)
    {
        $volontaire = NULL;
        if ($id) {
            $volontaire = $this->volontaireModel->where('userid', $id)->first();
        } else {
            $volontaire = new stdClass();
            $volontaire->id = 0;
            $volontaire->titre = "";
            $volontaire->nom = "";
            $volontaire->prenom = "";
            $volontaire->surnom = "";
            $volontaire->genre = "";
            $volontaire->email = "";
            $volontaire->telephone = "";
            $volontaire->dateNaissance = "";
            $volontaire->nationaliteId = "";
            $volontaire->professionId = "";
            $volontaire->fonction = "";
            $volontaire->employeur = "";
            $volontaire->estStaff = "";
            $volontaire->passportImagePath = "";
            $volontaire->brancheId = "";
            $volontaire->communeId = "";
            $volontaire->zoneId = "";
            $volontaire->collineId = "";
            $volontaire->estSectionJeunesse = "";
            $volontaire->sectionJeunesse = "";
        }

        return $volontaire;
    }

    private function valideForm()
    {
        $userId = \auth()->user()->id;

        $rules = [];
        $rules['nom']      = ['rules'  => 'required|min_length[2]|max_length[100]'];
        $rules['prenom']     = ['rules'  => 'required|min_length[2]|max_length[100]'];
        $rules['genre']     = ['rules'  => 'required|in_list[FEMME,HOMME,AUTRE,NON SPECIFIE]'];
        $rules['telephone']     = ['rules'  => 'required|exact_length[8]|is_natural|is_unique[vt_volontaires.telephone, userId,' . $userId . ']'];
        $rules['email']     = ['rules'  => 'required|valid_email|is_unique[vt_volontaires.email, userId,' . $userId . ']'];
        $rules['dateNaissance']     = ['rules'  => 'required'];
        $rules['professionId']     = ['rules'  => 'required'];
        $rules['nationaliteId']     = ['rules'  => 'required'];

        return $rules;
    }

    function pdf()
    {
        $pdf = new PdfModel();

        $volontaire = $this->volontaireModel->where('userId', \auth()->user()->id)->first();

        $nom = isset($volontaire->nom) ? $volontaire->nom : "-";
        $prenom = isset($volontaire->prenom) ? $volontaire->prenom : "-";
        $nationaliteId = isset($volontaire->nationaliteId) ? $volontaire->nationaliteId : 1000;
        $surnom = isset($volontaire->surnom) ? $volontaire->surnom : "-";
        $genre = isset($volontaire->genre) ? $volontaire->genre : "-";

        $pdf->AliasNbPages();
        $pdf->SetAutoPageBreak(1, 13);
        $pdf->AddPage();
        $this->response->setHeader('Content-Type', 'application/pdf');
        $pdf->SetFont('Times', '', 10);
        $pdf->Cell(185, 3, $this->formatText("Généré, le ") . date('d/m/Y'), 0, 1, 'R');
        $pdf->SetFont('Times', 'B', 14);
        $pdf->Cell(185, 20, $this->formatText($nom . " " . $prenom . "(" . CommonTrait::formatData(new Pays(), $nationaliteId) . ") - VOLONTAIRE"), "B", 1, 'C');
        $pdf->Ln(5);


        $pdf->SetFont('Times', 'B', 10);
        $pdf->Cell(40, 5, $this->formatText("Nom"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Prénoms "), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText("Surnom"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Genre"), '', 1, 'L');

        $pdf->SetFont('Times', '', 10);
        $pdf->Cell(40, 5, $this->formatText($nom), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($prenom), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText($surnom), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($genre), '', 1, 'L');
        $pdf->Ln(5);

        $pdf->SetFont('Times', 'B', 10);
        $pdf->Cell(40, 5, $this->formatText("Titre"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Téléphone"), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText("E-mail"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Né(e) le"), '', 1, 'L');

        $titre = isset($volontaire->titre) ? $volontaire->titre : "-";
        $telephone = isset($volontaire->telephone) ? $volontaire->telephone : "-";
        $email = isset($volontaire->email) ? $volontaire->email : "-";
        $dateNaissance = isset($volontaire->dateNaissance) ? $volontaire->dateNaissance : "-";

        $pdf->SetFont('Times', '', 10);
        $pdf->Cell(40, 5, $this->formatText($titre), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($telephone), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText($email), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($dateNaissance), '', 1, 'L');
        $pdf->Ln(5);

        $pdf->SetFont('Times', 'B', 10);
        $pdf->Cell(40, 5, $this->formatText("Branche"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Commune"), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText("Zone"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Colline"), '', 1, 'L');

        $branche = $volontaire->brancheId > 0 ? CommonTrait::formatData($this->brancheModel, $volontaire->brancheId) : "-";
        $commune = $volontaire->communeId > 0 ? CommonTrait::formatData(new CommuneModel(), $volontaire->communeId) : "-";
        $zone = $volontaire->zoneId > 0 ? CommonTrait::formatData(new ZoneModel(), $volontaire->zoneId) : "-";
        $colline = $volontaire->collineId > 0 ? CommonTrait::formatData(new CollineModel(), $volontaire->collineId) : "-";

        $pdf->SetFont('Times', '', 10);
        $pdf->Cell(40, 5, $this->formatText($branche), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($commune), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText($zone), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($colline), '', 1, 'L');
        $pdf->Ln(5);

        $pdf->SetFont('Times', 'B', 10);
        $pdf->Cell(40, 5, $this->formatText("Profession"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Est staff de la NS?"), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText("Employeur/Fonction"), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText("Profession"), '', 1, 'L');

        $profession = $volontaire->professionId > 0 ? CommonTrait::formatData($this->professionModel, $volontaire->professionId) : "-";
        $estStaff = isset($volontaire->estStaff) ? $volontaire->estStaff : "...";
        $profession = $volontaire->professionId > 0 ? CommonTrait::formatData($this->professionModel, $volontaire->professionId) : "-";

        $pdf->SetFont('Times', '', 10);
        $pdf->Cell(40, 5, $this->formatText($profession), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($estStaff), '', 0, 'L');
        $pdf->Cell(64, 5, $this->formatText($volontaire->employeur . "/" . $volontaire->fonction), '', 0, 'L');
        $pdf->Cell(40, 5, $this->formatText($profession), '', 1, 'L');
        $pdf->Ln(5);

        $fileName = $this->formatText("identite/" . $nom . "_" . $prenom . ".pdf");
        $pdf->Output($fileName, 'F');
        $pdf->Output();
    }

    function formatText($string = NULL)
    {
        if (gettype($string) == "string") {
            $string = UConverter::transcode($string, 'ISO-8859-1', 'UTF8');
        }

        if (gettype($string) == "array") {
            foreach ($string as $key => $value) {
                $string[$key] = UConverter::transcode($value, 'ISO-8859-1', 'UTF8');
            }
        }

        return $string;
    }

    function export()
    {
        $uri = current_url(true);
        $brancheId = $uri->getSegment(4);

        if ($brancheId > 0) {
            $branche = CommonTrait::formatData(new BrancheModel(), $brancheId);
        } else {
            $branche = "Toutes les branches";
        }

        $fileName = str_replace(' ', '', $branche);
        $csv_file_name = "rapportExport/volontaire/volontaire_" . $fileName . ".csv";
        $file = fopen($csv_file_name, 'w');

        $arrays = [];
        $arrays[0] = ['', $this->formatText("Les Volontaires")];
        $arrays[1] = ['', $this->formatText($branche)];
        $arrays[2] = [];
        $arrays[3] = [
            '#',
            $this->formatText('Volontaire'),
            $this->formatText('Branche'),
            $this->formatText('Commune'),
            $this->formatText("Genre"),
            $this->formatText("E-mail"),
            $this->formatText("telephone"),
            $this->formatText("Nee"),
            $this->formatText("Profession"),
            $this->formatText("Employeur"),
            $this->formatText("Appartien a une section"),
        ];

        if ($brancheId > 0) {
            $volontaires = $this->volontaireModel->where('brancheId', $brancheId)->findAll();
        } else {
            $volontaires = $this->volontaireModel->findAll();
        }

        $i = 1;
        foreach ($volontaires as $volontaire) {
            $sub_array = [];
            $sub_array[] = $i;
            $sub_array[] =  $volontaire->titre . " " . $volontaire->nom . " " . $volontaire->prenom;
            $sub_array[] =  $volontaire->brancheId > 0 ?$this->formatText( CommonTrait::formatData(new BrancheModel(), $volontaire->brancheId)) : "";
            $sub_array[] =  $volontaire->communeId > 0 ? $this->formatText(CommonTrait::formatData(new CommuneModel(), $volontaire->communeId)) : "";
            $sub_array[] =  $volontaire->genre;
            $sub_array[] =  $volontaire->email;
            $sub_array[] =  $volontaire->telephone;
            $sub_array[] =  $volontaire->dateNaissance;
            $sub_array[] = $volontaire->professionId > 0 ? $this->formatText(CommonTrait::formatData(new ProfessionModel(), $volontaire->professionId)):"";
            $sub_array[] =  $volontaire->employeur;
            $sub_array[] =  $this->formatText(CommonTrait::yesno('', $volontaire->estSectionJeunesse));

            $arrays[] = $sub_array;
            $i++;
        }

        foreach ($arrays as $array) {
            fputcsv($file, $array);
        }
        fclose($file);
        $btn = anchor($csv_file_name, "Exporter un <i class='fa fa-file-csv'></i>", ['class' => 'text-success', 'target' => '_blank']);

        $status = 0;
        if (fileExists($csv_file_name)) {
            $status = 1;
        }

        return $this->response->setJSON(['status' => $status, 'btnExport' => $btn]);
    }
}
