<?php

namespace Modules\Params\Controllers;

use App\Controllers\BaseController;
use App\Libraries\DatatableQuery;
use App\Traits\CommonTrait;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Modules\Params\Models\BrancheModel;
use Modules\Params\Models\CommuneModel;
use Psr\Log\LoggerInterface;
use stdClass;

class CommuneController extends BaseController
{
    public CommuneModel $communeModel;
    public BrancheModel $brancheModel;
    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);
        $this->communeModel = new CommuneModel();
        $this->brancheModel = new BrancheModel();
    }

    public function index()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(4);

        $commune =  $this->initCommune($id);

        $this->viewData['url_list'] = "/params/commune/data";
        $this->viewData['js'] = "";
        $this->viewData['route'] = "/params/commune/save";
        $this->viewData['commune'] =  $commune;
        $this->viewData['branches'] = CommonTrait::formatData($this->brancheModel);

        return view('params/commune/add', $this->viewData);
    }

    public function save()
    {
        $id = $this->request->getPost('id');

        if (!$this->validate($this->_rules())) {
            $commune =  $this->initCommune($id);

            $this->viewData['url_list'] = "/params/commune/data";
            $this->viewData['js'] = "";
            $this->viewData['route'] = "/params/commune/save";
            $this->viewData['commune'] =  $commune;
            $this->viewData['branches'] = CommonTrait::formatData($this->brancheModel);

            return view('params/commune/add', $this->viewData);
        } else {

            $uneCommune = $this->uneCommune();
            if ($id) {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>La branche <b>" . $uneCommune['description'] . "</b> a été modifiée</div>");
                $stmt = $this->communeModel->update($id, $uneCommune);
            } else {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>La branche <b>" . $uneCommune['description'] . "</b> a été bien enregistrée</div>");
                $stmt = $this->communeModel->save($uneCommune);
            }

            if ($stmt) {
                return redirect()->to(site_url($this->viewData['locale'] . '/params/commune'));
            }
        }
    }


    public function delete()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(5);

        $stmnt = $this->communeModel->delete($id);
        
        if ($stmnt) {
            $this->session->setFlashdata('message', "<div class='row alert alert-success'>La branche a été suprimée</div>");
            return redirect()->to(site_url($this->viewData['locale'] . '/params/commune'));
        }
        
    }

    function communes() {
        $uri = current_url(true);

        $brancheId = $uri->getSegment(4); 
        $communeId = $uri->getSegment(5);

        $output = ['communes' => $this->formatStringCommunes((new CommuneModel())->where("brancheId", $brancheId)->get()->getResult(), $communeId)];
        return $this->response->setJSON($output);
    }


    private function uneCommune()
    {
        $commune = [];
        $commune['description'] = $this->request->getPost('description');
        $commune['brancheId'] = $this->request->getPost('brancheId');
        $commune['longitude'] = $this->request->getPost('longitude');
        $commune['latitude'] = $this->request->getPost('latitude');

        return $commune;
    }

    private function initCommune($id = NULL)
    {
        $commune = NULL;
        if ($id) {
            $commune = $this->communeModel->find($id);
        } else {
            $commune = new stdClass();
            $commune->id = 0;
            $commune->brancheId = "";
            $commune->description = "";
            $commune->longitude = "-1.0";
            $commune->latitude = "-1.0";
        }

        return $commune;
    }

    private function _rules()
    {
        $rules = [];
        $rules['description'] = ['rules'  => 'required|min_length[2]|max_length[100]'];
        $rules['brancheId'] = ['rules'  => 'required|is_natural_no_zero'];
        $rules['longitude'] = ['rules'  => 'permit_empty|decimal'];
        $rules['latitude'] = ['rules'  => 'permit_empty|decimal'];

        return $rules;
    }



    function fecth()
    {
        $search_column = ['id', 'description', 'latitude', 'longitude'];
        $order_column = ['id', 'description', 'latitude', 'longitude'];

        $fetch_data = DatatableQuery::build_query($this->communeModel, $search_column, $order_column);
        $output = $this->formatCommunes($this->viewData['locale'], $fetch_data);

        return $this->response->setJSON($output);
    }

    private function formatCommunes($locale, $datas)
    {
        $i = 1;
        $array_data = array();
        if (!empty($datas['data'])) {
            foreach ($datas['data'] as $data) {

                $sub_array = array();

                $sub_array[] =  $i;
                $sub_array[] =  $data->description;
                $sub_array[] =  CommonTrait::formatData($this->brancheModel,$data->brancheId);
                $sub_array[] =  $data->latitude;
                $sub_array[] =  $data->longitude;
                $option = "-";
                if (\auth()->user()->inGroup('hr', 'admin') || (\auth()->user()->hasPermission('params.edit') && \auth()->user()->hasPermission('params.delete'))) {
                $option = anchor($locale . "/params/commune/" . $data->id, "<i class='fa fa-edit'></i>", ['class' => 'text-info']) . " " 
                        . anchor($locale . "/params/commune/delete/" . $data->id, "<i class='fa fa-trash'></i>", ['class' => 'text-danger']);
                }
                $sub_array[] = $option;

                $array_data[] = $sub_array;
                $i++;
            }
        }

        return [
            "draw" => intval($_POST["draw"]),
            "recordsTotal" => $datas['recordsTotal'],
            "recordsFiltered" => $datas['recordsFiltered'],
            "data" => $array_data
        ];
    }

    private function formatStringCommunes($communes, $communeId){        

        $options = "<option value='0'>Choix (commune)</option>";

        foreach ($communes as $commune) {
            # code...
            $selected = $commune->id == $communeId?"selected":"";
            $options .= "<option $selected value='".$commune->id."'>".$commune->description."</option>";
        }

        return $options;
    }
}
