<?php

namespace Modules\data\Controllers;

use App\Controllers\BaseController;
use App\Libraries\DatatableQuery;
use App\Traits\CommonTrait;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Modules\data\Models\InfrastructureModel;
use Modules\Params\Models\BrancheModel;
use Modules\Params\Models\CollineModel;
use Modules\Params\Models\CommuneModel;
use Modules\Params\Models\TypeAttestation;
use Modules\Params\Models\TypeInfrastructure;
use Modules\Params\Models\ZoneModel;
use Psr\Log\LoggerInterface;
use stdClass;
use UConverter;

use function PHPUnit\Framework\fileExists;


class InfrastructureController extends BaseController
{
    public InfrastructureModel $infrastructureModel;
    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);
        $this->infrastructureModel = new InfrastructureModel();
    }

    public function index()
    {
        $this->viewData['url_list'] = "/data/infrastructure/data";
        $this->viewData['js'] = "Common.js";
        $this->viewData['title'] = "Liste des infrastructures";
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());

        return view('data/infrastructure/index', $this->viewData);
    }

    public function add()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(5);

        $this->viewData['url_list'] = "";
        $this->viewData['js'] = "Volontaire.js";
        $this->viewData['route'] = "/data/infrastructure/save";
        $this->viewData['infrastructure'] =  $this->initInfrastructure($id);
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());
        $this->viewData['typeInfrastructures'] = CommonTrait::formatData(new TypeInfrastructure());
        $this->viewData['typeAttestations'] = CommonTrait::formatData(new TypeAttestation());
        $this->viewData['yesnon'] = CommonTrait::yesno();

        if ($id > 0)
            $title = "Editer un infrastructure";
        else
            $title = "Nouveau infrastructure";

        $this->viewData['title'] = $title;

        return view('data/infrastructure/add', $this->viewData);
    }

    public function save()
    {
        $id = $this->request->getPost('id');

        if (!$this->validate($this->_rules())) {

            $this->viewData['url_list'] = "/data/infrastructure/data";
            $this->viewData['js'] = "";
            $this->viewData['route'] = "/data/infrastructure/save";
            $this->viewData['infrastructure'] =  $this->initInfrastructure($id);
            $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());
            $this->viewData['typeInfrastructures'] = CommonTrait::formatData(new TypeInfrastructure());
            $this->viewData['typeAttestations'] = CommonTrait::formatData(new TypeAttestation());
            $this->viewData['yesnon'] = CommonTrait::yesno();

            if ($id > 0)
                $title = "Editer un infrastructure";
            else
                $title = "Nouveau infrastructure";

            $this->viewData['title'] = $title;

            return view('data/infrastructure/add', $this->viewData);
        } else {

            $uneInfrastructure = $this->uneInfrastructure();
            if ($id) {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>L'infrastructure a été modifié</div>");
                $stmt = $this->infrastructureModel->update($id, $uneInfrastructure);
            } else {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>L'infrastructure a été bien enregistré</div>");
                $stmt = $this->infrastructureModel->save($uneInfrastructure);
            }

            if ($stmt) {
                return redirect()->to(site_url($this->viewData['locale'] . '/data/infrastructure'));
            }
        }
    }


    public function delete()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(6);

        $stmnt = $this->infrastructureModel->delete($id);

        if ($stmnt) {
            $this->session->setFlashdata('message', "<div class='row alert alert-success'>L'infrastructure a été suprimé</div>");
            return redirect()->to(site_url($this->viewData['locale'] . '/data/infrastructure'));
        }
    }

    private function uneInfrastructure()
    {
        $infrastructure = [];
        $infrastructure['brancheId'] = $this->request->getPost('brancheId');
        $infrastructure['communeId'] = $this->request->getPost('communeId');
        $infrastructure['zoneId'] = $this->request->getPost('zoneId');
        $infrastructure['collineId'] = $this->request->getPost('collineId');
        $infrastructure['typeInfrastructureId'] = $this->request->getPost('typeInfrastructureId');
        $infrastructure['acquisitionProprieteImmo'] = $this->request->getPost('acquisitionProprieteImmo');
        $infrastructure['typeAttestationId'] = $this->request->getPost('typeAttestationId');
        $infrastructure['coutEstime'] = $this->request->getPost('coutEstime');
        $infrastructure['userId'] = \auth()->user()->id;

        return $infrastructure;
    }

    private function initInfrastructure($id = NULL)
    {
        $infrastructure = NULL;
        if ($id) {
            $infrastructure = $this->infrastructureModel->find($id);
        } else {
            $infrastructure = new stdClass();
            $infrastructure->id = 0;
            $infrastructure->brancheId = NULL;
            $infrastructure->communeId = NULL;
            $infrastructure->zoneId = NULL;
            $infrastructure->collineId = NULL;
            $infrastructure->typeInfrastructureId = '';
            $infrastructure->acquisitionProprieteImmo = NULL;
            $infrastructure->typeAttestationId = 0;
            $infrastructure->coutEstime = 0;
        }

        return $infrastructure;
    }

    private function _rules()
    {
        $rules = [];
        $rules['brancheId'] = ['rules'  => 'required'];
        $rules['communeId'] = ['rules'  => 'required'];
        $rules['zoneId'] = ['rules'  => 'required'];
        $rules['collineId'] = ['rules'  => 'required'];
        $rules['typeInfrastructureId'] = ['rules'  => 'required'];
        $rules['acquisitionProprieteImmo'] = ['rules'  => 'required'];
        $rules['typeAttestationId'] = ['rules'  => 'required'];
        $rules['coutEstime'] = ['rules'  => 'required'];

        return $rules;
    }

    function fecth()
    {
        $search_column = ['brancheId', 'communeId', 'zoneId', 'collineId', 'typeInfrastructureId', 'acquisitionProprieteImmo', 'typeAttestationId', 'coutEstime'];
        $order_column = ['brancheId', 'communeId', 'zoneId', 'collineId', 'typeInfrastructureId', 'acquisitionProprieteImmo', 'typeAttestationId', 'coutEstime'];

        $infrastructureMdl = $this->infrastructureModel;
        if (\auth()->user()->inGroup('brancheManager')) {
            $infrastructureMdl->where('brancheId', \auth()->user()->brancheId);
        }

        if (\auth()->user()->inGroup('communeManager')) {
            $infrastructureMdl->where('communeId', \auth()->user()->communeId);
        }

        if ($this->request->getPost('brancheId') > 0) {
            $infrastructureMdl->where('brancheId', $this->request->getPost('brancheId'));
        }

        $fetch_data = DatatableQuery::build_query($infrastructureMdl, $search_column, $order_column);
        $output = $this->formatInfrastructure($this->viewData['locale'], $fetch_data);

        return $this->response->setJSON($output);
    }

    private function formatInfrastructure($locale, $datas)
    {
        $i = 1;
        $array_data = array();
        if (!empty($datas['data'])) {
            foreach ($datas['data'] as $data) {

                $sub_array = array();

                $sub_array[] =  $i;
                $sub_array[] =  CommonTrait::formatData(new BrancheModel(), $data->brancheId);
                $sub_array[] =  CommonTrait::formatData(new CommuneModel(), $data->communeId);
                $sub_array[] =  CommonTrait::formatData(new TypeInfrastructure(), $data->typeInfrastructureId);
                $sub_array[] =  CommonTrait::yesno(NULL, $data->acquisitionProprieteImmo);
                $sub_array[] =  CommonTrait::formatData(new TypeAttestation(), $data->typeAttestationId);
                $sub_array[] =  $data->coutEstime;

                $option = "-";
                if (\auth()->user()->inGroup('hr', 'admin') || (\auth()->user()->hasPermission('params.edit') && \auth()->user()->hasPermission('params.delete'))) {
                    $option = anchor($locale . "/data/infrastructure/add/" . $data->id, "<i class='fa fa-edit'></i>", ['class' => 'text-info']) . " "
                        . anchor($locale . "/data/infrastructure/delete/" . $data->id, "<i class='fa fa-trash'></i>", ['class' => 'text-danger']);
                }
                $sub_array[] = $option;

                $array_data[] = $sub_array;
                $i++;
            }
        }

        return [
            "draw" => intval($_POST["draw"]),
            "recordsTotal" => $datas['recordsTotal'],
            "recordsFiltered" => $datas['recordsFiltered'],
            "data" => $array_data
        ];
    }
    function export()
    {
        $uri = current_url(true);
        $brancheId = $uri->getSegment(5);
       

        if ($brancheId > 0) {
            $branche = CommonTrait::formatData(new BrancheModel(), $brancheId);
        } else {
            $branche = "Toutes les branches";
        }

        $fileName = str_replace(' ', '', $branche);
        $csv_file_name = "rapportExport/infrastructure/infrastructure_" . $fileName . ".csv";
        $file = fopen($csv_file_name, 'w');

        $arrays = [];
        $arrays[0] = ['', $this->formatText("Infrastructures immobilières")];
        $arrays[1] = ['', $this->formatText($branche)];
        $arrays[2] = [];
        $arrays[3] = [
            '#',
            $this->formatText('Province'),
            $this->formatText('Branche'),
            $this->formatText('Zone/CPEV'),
            $this->formatText('Colline'),
            $this->formatText('Type d\'infrastructure'),
            $this->formatText('Titre acquis'),
            $this->formatText('Type attestation'),
            $this->formatText("Coût estimé"),
        ];
        
        if ($brancheId > 0) {
            $infrastructures = $this->infrastructureModel->where('brancheId', $brancheId)->findAll();
        } else {
            $infrastructures = $this->infrastructureModel->findAll();
        }

        
        $i = 1;
        foreach ($infrastructures as $infrastructure) {
            $sub_array = [];
            $sub_array[] = $i;
            $sub_array[] =  $this->formatText($infrastructure->brancheId > 0 ? CommonTrait::formatData(new BrancheModel(), $infrastructure->brancheId):"");
            $sub_array[] =  $this->formatText($infrastructure->communeId > 0 ? CommonTrait::formatData(new CommuneModel(), $infrastructure->communeId):"");
            $sub_array[] =  $this->formatText($infrastructure->zoneId > 0 ? CommonTrait::formatData(new ZoneModel(), $infrastructure->zoneId):"");
            $sub_array[] =  $this->formatText($infrastructure->collineId > 0 ? CommonTrait::formatData(new CollineModel(), $infrastructure->collineId):"");
            $sub_array[] =  $this->formatText($infrastructure->typeInfrastructureId > 0? CommonTrait::formatData(new TypeInfrastructure(), $infrastructure->typeInfrastructureId):"");
            $sub_array[] =  CommonTrait::yesno(NULL, $infrastructure->acquisitionProprieteImmo);
            $sub_array[] =  CommonTrait::formatData(new TypeAttestation(), $infrastructure->typeAttestationId);
            $sub_array[] =  $infrastructure->coutEstime;


            $arrays[] = $sub_array;
            $i++;
        }


        foreach ($arrays as $array) {
            fputcsv($file, $array, ";");
        }
        fclose($file);
        $btn = anchor($csv_file_name, "Exporter un <i class='fa fa-file-csv'></i>", ['class' => 'text-success', 'target' => '_blank']);

        $status = 0;
        if (FileExists($csv_file_name)) {
            $status = 1;
        }

       return $this->response->setJSON(['status' => $status, 'btnExport' => $btn]);
    }

    function formatText($string = "")
    {
        return UConverter::transcode($string, 'ISO-8859-1', 'UTF8');
    }
}
